<?php

namespace App\Services;

use App\Models\NumerologyCalculation;
use App\Models\User;

class NumerologyService
{
    public function __construct(
        private NumerologyCalculationService $calculationService,
        private SWOTCombinationService $swotService,
        private RootInterpretationService $interpretationService,
        private BusinessGuidanceService $businessGuidanceService
    ) {}

    public function performFullCalculation(string $name, string $birthdate, ?string $industry = null): array
    {
        // Perform basic calculation
        $calculation = $this->calculationService->calculateNumerology($name, $birthdate, $industry);
        
        // Find SWOT combinations
        $swotCombinations = $this->swotService->findSWOTCombinations(
            $calculation['L6'],
            $calculation['L7'],
            $calculation['L8'],
            $calculation['L9']
        );

        // Get root number interpretation
        $interpretation = $this->interpretationService->getInterpretation($calculation['root_number']);

        // Get business guidance based on element
        $businessGuidance = null;
        if ($interpretation) {
            $businessGuidance = $this->businessGuidanceService->getBusinessGuidance(
                $interpretation['element'],
                $industry
            );
        }

        // Get learning style
        $learningStyle = $this->businessGuidanceService->getLearningStyle($calculation['root_number']);

        return [
            'calculation' => $calculation,
            'interpretation' => $interpretation,
            'swot_combinations' => $swotCombinations,
            'business_guidance' => $businessGuidance,
            'learning_style' => $learningStyle,
        ];
    }

    public function saveCalculation(array $calculationData, ?User $user = null): NumerologyCalculation
    {
        $data = [
            'user_id' => $user?->id,
            'name' => $calculationData['calculation']['name'],
            'birthdate' => $calculationData['calculation']['birthdate'],
            'industry' => $calculationData['calculation']['industry'],
            'L5' => $calculationData['calculation']['L5'],
            'L6' => $calculationData['calculation']['L6'],
            'L7' => $calculationData['calculation']['L7'],
            'L8' => $calculationData['calculation']['L8'],
            'L9' => $calculationData['calculation']['L9'],
            'root_number' => $calculationData['calculation']['root_number'],
            'swot_combinations' => $calculationData['swot_combinations'],
        ];

        return NumerologyCalculation::create($data);
    }

    public function getUserCalculations(User $user): array
    {
        return $user->numerologyCalculations()
            ->orderBy('created_at', 'desc')
            ->get()
            ->toArray();
    }

    public function deleteCalculation(int $calculationId, User $user): bool
    {
        $calculation = $user->numerologyCalculations()->find($calculationId);
        
        if (!$calculation) {
            return false;
        }

        return $calculation->delete();
    }
}