# Numerology Pyramid Calculator - Laravel React Development

## PROJECT SETUP
- Laravel backend API serving at http://swottalent.test (using Laravel Herd)
- React frontend (can be separate port or integrated)
- Database: MySQL for storing calculations and user data
- Authentication: Laravel Sanctum for API authentication

## CORE FUNCTIONALITY

### 1. NUMEROLOGY CALCULATION ENGINE (Backend - Laravel)
Create a Laravel service class that implements these exact calculation rules:

**Input Processing:**
- Parse flexible date formats: "31-July-1985", "31/07/1985", "31 July 1985"
- Handle month names in both English and Malay
- Validate dates (1800-2099 year range)

**Pyramid Calculations (Lines 5-9 only):**
```php
// L5: Raw birthdate components
L5 = [day, month, year_first_two_digits, year_last_two_digits]

// L6: Digit reduction with special rule
L6 = [
    reduce(L5[0]),
    reduce(L5[1]), 
    reduce(L5[2]),
    L5[3] == 0 ? 5 : reduce(L5[3])  // Special case for years ending in '00'
]

// L7: Pairwise sums
L7 = [reduce(L6[0] + L6[1]), reduce(L6[2] + L6[3])]

// L8: Complex calculation (7 elements)
A = reduce(L6[1] + L7[0])
B = reduce(L6[0] + L7[0]) 
C = reduce(A + B)
D = reduce(L6[2] + L7[1])
E = reduce(L6[3] + L7[1])
F = reduce(D + E)
G = reduce(L7[0] + L7[1])
L8 = [C, B, A, G, D, E, F]

// L9: Final calculation
Left = reduce(L7[1] + G)
Right = reduce(L7[0] + G)
Middle = reduce(Left + Right)
L9 = [Left, Middle, Right]

// Root Number = G (middle of L8)
```

**SWOT Combinations Detection:**
- Scan L6-L9 grid for adjacent 2-3 digit sequences (8-direction adjacency)
- Match only against this official list from reference material:

```php
$swotCombinations = [
    '123' => ['label' => 'S', 'description' => 'Kekuatan pada suara (penyanyi,guru, sales, marketing, usahawan)'],
    '426' => ['label' => 'S', 'description' => 'Pandai syarahan / ceramah / pendidikan'],
    '246' => ['label' => 'S', 'description' => 'Pandai syarahan / ceramah / pendidikan'],
    '729' => ['label' => 'S', 'description' => 'Sangat menawan dan mudah yakin/tarikan orang, karisma "Pelaris"'],
    '279' => ['label' => 'S', 'description' => 'Sangat menawan dan mudah yakin/tarikan orang, karisma "Pelaris"'],
    '573' => ['label' => 'W', 'description' => 'Berpengaruh, mudah bergaul dengan kenamaan dan tertinggi/ koprat atau orang atas duit habis sebab di ambil orang atau berbelanja untuk orang kenamaan atau orang atas. Mudah disabotaj orang.'],
    '56' => ['label' => 'S', 'description' => 'Sangat berhati-hati dengan wang (sesuai jadi pegawai kewangan)'],
    '666' => ['label' => 'S', 'description' => 'Sangat berhati-hati dengan wang (sesuai jadi pegawai kewangan)'],
    '966' => ['label' => 'O', 'description' => 'Mudah untung (bagus jadi usahawan)'],
    '336' => ['label' => 'T', 'description' => 'Positif, pergerakan wang yang aktif Negatif, boros kerana duit "panas"'],
    '663' => ['label' => 'W', 'description' => 'Aliran wang tunai terlalu cepat'],
    '393' => ['label' => 'T', 'description' => 'Kejayaan yang singkat (positif-cepat berjaya negatif- tak bertahan lama)'],
    '933' => ['label' => 'T', 'description' => 'Kejayaan yang singkat (positif-cepat berjaya negatif- tak bertahan lama)'],
    '119' => ['label' => 'W', 'description' => 'Sangat berdikari, tidak boleh jadi rakan niaga, suka conquer, Rasa kesunyian dan sendirian'],
    '191' => ['label' => 'W', 'description' => 'Sangat berdikari, tidak boleh jadi rakan niaga, suka conquer, Rasa kesunyian dan sendirian'],
    '843' => ['label' => 'O', 'description' => 'Selalu tercari-cari kejayaan, mudah bosan ,tamakkan peluang, sentiasa ingin tukar citarasa maka duit habis. Terdapat pada orang MLM. Banyak idea dan inovatif.'],
    '456' => ['label' => 'T', 'description' => 'Terlibat/bertanggungjawab di dalam aliran wang (mudah untuk terlibat dengan rasuah jika lalai.'],
    '111' => ['label' => 'T', 'description' => 'Mudah depressi/sakit jiwa/tekanan emosi'],
    '6666' => ['label' => 'T', 'description' => 'Impak dengan besi (kemalangan,pembedahan)'],
    '222' => ['label' => 'S', 'description' => 'Dikelilingi wanita , dibiayai wanita, mudah rapat dengan wanita'],
    '777' => ['label' => 'S', 'description' => 'Dikelilingi lelaki (jiwa wanita)/ Network kuat'],
    '999' => ['label' => 'O', 'description' => 'Banyak peluang dan kejayaan tetapi kesunyian suka tinggal dikawasan bukit berhutan'],
    '888' => ['label' => 'T', 'description' => 'Terlalu "panas"- Hiperaktif'],
    '333' => ['label' => 'T', 'description' => 'Terlalu "panas"- Agresif, pendendam'],
    '11' => ['label' => 'W', 'description' => 'Sangat berdikari, namun"ditikam belakang" Berjauhan (mental/fizikal) boleh berpisah/bercerai'],
    '459' => ['label' => 'O', 'description' => 'Mempunyai gerak hati atau muslihat, ilham, dunia kecantikan & Kesihatan'],
    '944' => ['label' => 'S', 'description' => 'Pengasas / Penerbitan / Rekacipta (Masakan, Fesyen), Herba, Kecantikan, pertanian, dunia akademik.'],
    '235' => ['label' => 'W', 'description' => 'Beremosi, mudah terusik, baran'],
    '786' => ['label' => 'O', 'description' => 'Orang yang suka beri tanggungjawab (kelebihan dan mudah rezekinya'],
    '156' => ['label' => 'S', 'description' => 'Bekerja diluar (outstation )'],
    '189' => ['label' => 'S', 'description' => 'Bergerak sana sini, tidak duduk satu tempat'],
    '83' => ['label' => 'W', 'description' => 'Selalu keluar dan tiada dirumah / di pejabat'],
    '71' => ['label' => 'W', 'description' => 'Mudah ditipu orang/ terlalu mudah percayakan orang'],
    '73' => ['label' => 'W', 'description' => 'Orang mudah datang dan pergi/ orang mudah takut dengan anda'],
    '37' => ['label' => 'W', 'description' => 'Orang mudah datang dan pergi/ orang mudah takut dengan anda'],
    '67' => ['label' => 'W', 'description' => 'Mudah kasihan kepada orang / mudah belanja orang']
];
```

### 2. ROOT NUMBER INTERPRETATIONS
Create a Laravel model/service with complete interpretations for Root Numbers 1-9:

```php
$rootInterpretations = [
    1 => [
        'title' => 'SENDIRI (Independent Leader)',
        'traits' => 'Bijak, natural born leader/pemimpin, kreatif, banyak idea baru, pandai analisa orang (baca orang), Berani, berdikari, positif, individualistic, agresif, perfectionist, keras kepala, sentiasa nak menang, ego, sangat dominan dalam perhubungan, dingin, sangat sensitive, sentiasa rasa betul, expectation yang tinggi, berterus terang.',
        'element' => 'LOGAM/BESI',
        'strengths' => 'Pemimpin semulajadi, keyakinan diri yang tinggi, bercita-cita tinggi',
        'challenges' => 'Terlalu mengongkong, pendiam, sangat spontan',
        'desires' => 'Mahu perhatian orang ramai, mencuba idea baru',
        'career' => 'Ahli politik, artis, pemilik perniagaan'
    ],
    2 => [
        'title' => 'SUARA (Voice/Harmony)',
        'traits' => 'Lembut, pembersih, jujur, tenang, dengar kata tapi tidak semestinya buat, memahami, sabar, mudah untuk berkawan, boleh diajak berbincang/diplomasi, suka bersimpati, boleh dipercayai, mudah memaafkan, pemalu, terlampau sensitive, depressi, tiada cita-cita, kurang bijak, sukar untuk buat keputusan dan tidak tetap dalam membuat keputusan, seseorang emotional.',
        'element' => 'AIR',
        'strengths' => 'Penyayang pendiam, analitikal, rakan yang ideal',
        'challenges' => 'Tidak mahu maju atau bersuara, berdiam diri, tidak suka butiran kecil, kesunyian',
        'desires' => 'Mahu hubungan, sempurna, keharmonian dan kemas',
        'career' => 'Diplomat kaunselor, rakan niaga'
    ],
    3 => [
        'title' => 'AKTIF (Active/Creative)',
        'traits' => 'Menawan semasa muda, menarik, aktif, cepat menyiapkan tugas tapi tidak sempurna, baran agresif, kurang tegas, perhubungan rapuh, mudah berkawan, bernafsu tinggi, suka mengikut trend, seorang yang mudah teruja, bersemangat, kreatif, artistic, egois, suka mengawal kritikal.',
        'element' => 'API',
        'strengths' => 'Ceria, peramah, imaginasi, bersungguh-sungguh',
        'challenges' => 'Tidak suka kerja rutin, kurang disiplin, kurang berjaya',
        'desires' => 'Menggerakkan tenaga dan idea, mahu keseronokan hidup',
        'career' => 'Setiausaha, perancang majlis, promoter bisnes'
    ],
    4 => [
        'title' => 'PERANCANG/ILMU (Planner/Knowledge)',
        'traits' => 'Kemahiran yang tinggi, bijak tidak mudah putus asa, perhubungan yang mudah putus, sukar simpan wang, bangga, suka bersangka buruk pada orang lain, konservatif, rajin, keras kepala, kedekut, praktikal, setia, logikal, suka pada benda rutin, tetap, mahukan sekuriti, tidak berani ambil risiko.',
        'element' => 'KAYU',
        'strengths' => 'Kerja keras, praktikal, siapkan kerja',
        'challenges' => 'Takut diasingkan, tiada rutin biasa, tidak fleksibal',
        'desires' => 'Mahu keselamatan, cipta undang-undang',
        'career' => 'Pengurus, penyokong pasukan belakang tabir'
    ],
    5 => [
        'title' => 'MENCARI/ARAH (Seeker/Direction)',
        'traits' => 'Pemimpin, berpangkat tinggi, kerja keras, seorang yang degil, adventurous, bertenaga, bernafsu tinggi, bercita rasa tinggi, bijak sana, charming, berfikiran jauh, berani mengambil risiko, suka/minat berniaga, boros suka shopping dan kuat berbelanja, moody, tak consistent, jealousy, seksi, berkira, suka bertangguh.',
        'element' => 'TANAH',
        'strengths' => 'Sangat berani dan mudah meyakinkan',
        'challenges' => 'Mudah bosan, resah, mudah dipengaruhi',
        'desires' => 'Cari peluang baru, berani ambil risiko, mahu cuba segalanya',
        'career' => 'Perhatian Umum, media, mencipta idea-idea baru'
    ],
    6 => [
        'title' => 'WANG (Wealth/Management)',
        'traits' => 'Citarasa mewah, kawal wang, berdikari, mahu menguasai tidak mementingkan diri sendiri, pendamai, charming, suka duduk rumah, bijak, menegakkan keadilan, berperikemanusiaan, jujur, penyayang, mengambil berat, tamak.',
        'element' => 'LOGAM/BESI',
        'strengths' => 'Penyayang, pengasuh, suka berkeluarga, boleh diharap',
        'challenges' => 'Rasa rendah diri, mudah semput pemikiran',
        'desires' => 'Ambil berat tentang orang tersayang, keselamatan dan keselesaan',
        'career' => 'Pembantu peribadi, pendidik, penyelia'
    ],
    7 => [
        'title' => 'KAWAN/PENGIKUT (Friend/Follower)',
        'traits' => 'Cerewet, bertelagah, popular, berdikari, tarik penyokong, mistik, berfalsafah, artistic, pendendam, kesunyian, moody, depressi, mengira, pemalas, suka mengasingkan diri, suka berfikir, analitikal, kekuatan kerohanian.',
        'element' => 'AIR',
        'strengths' => 'Pemikir yang hebat, berfalsafah, unik, pelik',
        'challenges' => 'Kurang peramah, takut tidak capai tahap hidup yang tinggi, kesunyian',
        'desires' => 'Mencari jawapan persoalan hidup, memerhati, menyelidik',
        'career' => 'Bidang analitikal kerja sendiri'
    ],
    8 => [
        'title' => 'TANGGUNGJAWAB & TEKANAN (Responsibility & Pressure)',
        'traits' => 'Pendiam tapi ramai kawan, bertanggungjawab, dipercayai, pertahankan kawan, suka kepada kebebasan dan kuasa, keras kepala, bersifat kepimpinan, suka terurus, suka membantu, baik hati, memahami, ambitious, gila wang, egois, dictator, mudah tertekan, 6th sense yang kuat.',
        'element' => 'API',
        'strengths' => 'Berpengaruh, berbakat, mendesak, bagus dengan wang',
        'challenges' => 'Kurang senang dengan orang yang menghalangnya',
        'desires' => 'Berusaha gigih, mahu mendapat kuasa dan status',
        'career' => 'Bidang professional, pengurus bisnes francais'
    ],
    9 => [
        'title' => 'KEJAYAAN (Success)',
        'traits' => 'Seorang yang bergaya, kemas, pandai PR (Public Relation), tarik simpati orang tua, kesunyian, bijak, suka memberi, passionate, berfalsafah, sensitive, ikhlas, berbakat, romantis, bersifat kepimpinan, memahami, emosi yang berlebihan, penyayang.',
        'element' => 'KAYU',
        'strengths' => 'Bersemangat, cergas, kreatif, mahu berbakti kepada dunia',
        'challenges' => 'Elakkan sifat buruk dan perhatikan butiran hidup',
        'desires' => 'Mahu melihat dunia, mencapai impian',
        'career' => 'Ketua masyarakat, perancang bisnes'
    ]
];
```

### 3. DATABASE SCHEMA
Create migrations for:

```sql
-- users table (Laravel default)
CREATE TABLE users (
    id BIGINT PRIMARY KEY,
    name VARCHAR(255),
    email VARCHAR(255) UNIQUE,
    email_verified_at TIMESTAMP NULL,
    password VARCHAR(255),
    remember_token VARCHAR(100) NULL,
    created_at TIMESTAMP,
    updated_at TIMESTAMP
);

-- numerology_calculations table
CREATE TABLE numerology_calculations (
    id BIGINT PRIMARY KEY,
    user_id BIGINT NULL,
    name VARCHAR(255) NOT NULL,
    birthdate DATE NOT NULL,
    industry VARCHAR(255) NULL,
    L5 JSON NOT NULL,
    L6 JSON NOT NULL,
    L7 JSON NOT NULL,
    L8 JSON NOT NULL,
    L9 JSON NOT NULL,
    root_number INT NOT NULL,
    swot_combinations JSON NULL,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL
);

-- user_profiles table (optional for additional user data)
CREATE TABLE user_profiles (
    id BIGINT PRIMARY KEY,
    user_id BIGINT UNIQUE,
    preferred_language ENUM('en', 'ms') DEFAULT 'ms',
    default_industry VARCHAR(255) NULL,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);
```

### 4. API ENDPOINTS (Laravel Routes)

```php
// routes/api.php
Route::post('/calculate-numerology', [NumerologyController::class, 'calculate']);
Route::get('/interpretations/{root_number}', [NumerologyController::class, 'getInterpretation']);

// Protected routes (require authentication)
Route::middleware('auth:sanctum')->group(function () {
    Route::post('/save-calculation', [NumerologyController::class, 'saveCalculation']);
    Route::get('/my-calculations', [NumerologyController::class, 'getUserCalculations']);
    Route::delete('/calculations/{id}', [NumerologyController::class, 'deleteCalculation']);
    Route::get('/profile', [UserController::class, 'getProfile']);
    Route::put('/profile', [UserController::class, 'updateProfile']);
});

// Authentication routes
Route::post('/register', [AuthController::class, 'register']);
Route::post('/login', [AuthController::class, 'login']);
Route::middleware('auth:sanctum')->post('/logout', [AuthController::class, 'logout']);
```

**API Response Formats:**

```php
// POST /api/calculate-numerology
// Request: { "name": "Ricky", "birthdate": "31-July-1985", "industry": "Sales" }
// Response:
{
    "success": true,
    "data": {
        "calculation": {
            "name": "Ricky",
            "birthdate": "1985-07-31",
            "industry": "Sales",
            "L5": [31, 7, 19, 85],
            "L6": [4, 7, 1, 4],
            "L7": [2, 5],
            "L8": [9, 6, 9, 7, 6, 9, 6],
            "L9": [3, 6, 9],
            "root_number": 7
        },
        "interpretation": {
            "title": "KAWAN/PENGIKUT (Friend/Follower)",
            "traits": "...",
            "element": "AIR",
            "strengths": "...",
            "challenges": "...",
            "career": "..."
        },
        "swot_combinations": [
            {
                "sequence": "967",
                "base_combo": "679",
                "label": "S",
                "description": "...",
                "cells": [[2,1], [2,2], [2,3]]
            }
        ],
        "business_guidance": {
            "element_category": "Water/Service",
            "suitable_businesses": "...",
            "industry_specific": "..."
        }
    }
}
```

### 5. REACT FRONTEND COMPONENTS

**Main Components Structure:**
```
src/
├── components/
│   ├── Layout/
│   │   ├── Header.jsx
│   │   ├── Footer.jsx
│   │   └── Navigation.jsx
│   ├── Numerology/
│   │   ├── NumerologyForm.jsx
│   │   ├── PyramidGrid.jsx
│   │   ├── RootAnalysis.jsx
│   │   ├── SWOTAnalysis.jsx
│   │   ├── BusinessGuidance.jsx
│   │   ├── LearningStyle.jsx
│   │   └── CalculationResults.jsx
│   ├── Auth/
│   │   ├── LoginForm.jsx
│   │   ├── RegisterForm.jsx
│   │   └── ProtectedRoute.jsx
│   └── Common/
│       ├── LoadingSpinner.jsx
│       ├── ErrorMessage.jsx
│       └── ConfirmDialog.jsx
├── pages/
│   ├── Home.jsx
│   ├── Calculator.jsx
│   ├── MyCalculations.jsx
│   ├── Profile.jsx
│   └── About.jsx
├── hooks/
│   ├── useAuth.js
│   ├── useNumerology.js
│   └── useLocalStorage.js
├── services/
│   ├── api.js
│   ├── auth.js
│   └── numerology.js
└── utils/
    ├── dateParser.js
    ├── validation.js
    └── constants.js
```

**NumerologyForm.jsx requirements:**
```jsx
import React, { useState } from 'react';
import { validateDate, validateName } from '../utils/validation';

const NumerologyForm = ({ onSubmit, loading }) => {
    const [formData, setFormData] = useState({
        name: '',
        birthdate: '',
        industry: ''
    });
    const [errors, setErrors] = useState({});

    // Form validation, submission logic
    // Support for flexible date formats
    // Loading states and error handling
    // Auto-save to localStorage
    
    return (
        <form className="bg-white rounded-xl shadow-lg p-8">
            {/* Form fields with validation */}
        </form>
    );
};
```

**PyramidGrid.jsx requirements:**
```jsx
const PyramidGrid = ({ calculation }) => {
    const { L5, L6, L7, L8, L9, root_number } = calculation;
    
    return (
        <div className="pyramid-container">
            <div className="grid grid-cols-7 gap-1 max-w-2xl mx-auto">
                {/* 7-column responsive grid layout */}
                {/* Highlight root number (L8[3]) */}
                {/* Empty cells for spacing */}
            </div>
            <div className="calculation-summary mt-4">
                {/* Show L5-L9 values */}
            </div>
        </div>
    );
};
```

**SWOTAnalysis.jsx requirements:**
```jsx
const SWOTAnalysis = ({ combinations }) => {
    const groupedCombinations = {
        S: combinations.filter(c => c.label === 'S'),
        W: combinations.filter(c => c.label === 'W'),
        O: combinations.filter(c => c.label === 'O'),
        T: combinations.filter(c => c.label === 'T')
    };

    const categoryColors = {
        S: 'bg-green-100 border-green-500',
        W: 'bg-red-100 border-red-500',
        O: 'bg-blue-100 border-blue-500',
        T: 'bg-yellow-100 border-yellow-500'
    };

    return (
        <div className="swot-analysis">
            {/* Grid layout for SWOT categories */}
            {/* Color-coded cards */}
            {/* Handle empty states */}
        </div>
    );
};
```

### 6. STYLING REQUIREMENTS
- **Framework**: Tailwind CSS
- **Theme**: Purple/blue gradient (`bg-gradient-to-br from-purple-600 to-blue-600`)
- **Cards**: White backgrounds with shadow (`bg-white shadow-lg rounded-xl`)
- **Mobile-first**: Responsive design for all screen sizes
- **Animations**: Smooth transitions (`transition-all duration-300`)
- **Typography**: Modern, readable fonts
- **Color Palette**:
  - Primary: Purple (#8B5CF6)
  - Secondary: Blue (#3B82F6)
  - Success: Green (#10B981)
  - Warning: Yellow (#F59E0B)
  - Danger: Red (#EF4444)

### 7. TECHNICAL REQUIREMENTS

**Backend (Laravel):**
- Laravel 10+ with PHP 8.1+
- MySQL 8.0+ database
- Laravel Sanctum for API authentication
- Form Request validation
- Resource controllers and API resources
- Eloquent ORM with proper relationships
- Database migrations and seeders
- Unit and Feature tests

**Frontend (React):**
- React 18+ with functional components
- React Router for navigation
- Axios for API calls
- Context API or Zustand for state management
- React Hook Form for form handling
- React Query for data fetching and caching
- Tailwind CSS for styling
- Vite for build tooling

**Development Tools:**
- ESLint and Prettier for code formatting
- Laravel Pint for PHP code style
- PHPUnit for backend testing
- Jest/RTL for frontend testing

### 8. BUSINESS LOGIC REQUIREMENTS

**Element-based Business Categories:**
```php
$businessCategories = [
    'LOGAM/BESI' => [
        'category' => 'Metal/Finance',
        'businesses' => 'Kewangan, Perbankan, Komponen Elektronik, Kejuruteraan, peralatan besi, guaman, seni silat, barangan kemas/aksesori, dunia automotif, Pengangkutan/logistik/pelancongan, Pegawai Kerajaan, Polis, Tentera, Penguatkuasa'
    ],
    'AIR' => [
        'category' => 'Water/Service', 
        'businesses' => 'Dunia komunikasi, barangan pengguna, Pemborong Eksport Import, Kimia, Perkapalan, logistik laluan laut, Hotel, Perkhidmatan, Dobi, Makanan laut, penternakan haiwan akuatik'
    ],
    'API' => [
        'category' => 'Fire/Energy',
        'businesses' => 'Dunia lakonan/hiburan, Pengiklanan, Kecantikan/Salun/Kosmetik, Dunia Janakuasa/Tenaga, Dunia Pencahayaan/Laser, Elektronik, Hiasan Dalaman, Psikologi, Restoran, Telekomunikasi, Komputer, Minyak dan Gas'
    ],
    'KAYU' => [
        'category' => 'Wood/Education',
        'businesses' => 'Pertanian/perladangan, Kesenian, Penyelidikan, Pendidikan, Fesyen, Perubatan, Kedai buku, Perpustakaan, muzik, Penulisan, Percetakan dan Penerbitan, Kilang kotak, Perkhidmatan masyarakat, Fabrik, Tekstil, Herba'
    ],
    'TANAH' => [
        'category' => 'Earth/Leadership',
        'businesses' => 'Arkitek, Kontraktor, Kejuruteraan sivil, Barangan Tembikar/ceramik/Jubin, Perlombongan, Hartanah, Kuari, Konsultansi, Pentadbiran, Politik'
    ]
];
```

**Learning Style Mapping:**
```php
$learningStyles = [
    1 => ['style' => 'KINESTETIK (BESI)', 'approach' => 'Hands-on learning, practical application, direct experience'],
    2 => ['style' => 'VISUAL (AIR)', 'approach' => 'Visual aids, charts, diagrams, written instructions'],
    3 => ['style' => 'AUDIO (API)', 'approach' => 'Verbal discussions, music, sound-based learning'],
    4 => ['style' => 'DIGITAL (KAYU)', 'approach' => 'Structured information, step-by-step processes, logical sequences'],
    5 => ['style' => 'FLEXIBLE (TANAH)', 'approach' => 'Varied approaches, leadership roles, dynamic environments'],
    6 => ['style' => 'KINESTETIK (BESI)', 'approach' => 'Hands-on learning, practical application, direct experience'],
    7 => ['style' => 'VISUAL (AIR)', 'approach' => 'Visual aids, charts, diagrams, written instructions'],
    8 => ['style' => 'AUDIO (API)', 'approach' => 'Verbal discussions, music, sound-based learning'],
    9 => ['style' => 'DIGITAL (KAYU)', 'approach' => 'Structured information, step-by-step processes, logical sequences']
];
```

### 9. DEVELOPMENT TASKS

**Phase 1: Backend Foundation**
1. Set up Laravel project with proper structure
2. Create database migrations and models
3. Implement numerology calculation service class
4. Build SWOT combination detection algorithm
5. Create API routes and controllers
6. Set up authentication with Laravel Sanctum
7. Write unit tests for calculation logic

**Phase 2: Frontend Setup**
1. Set up React application with Vite
2. Configure Tailwind CSS
3. Set up React Router for navigation
4. Create basic component structure
5. Implement API service layer
6. Set up state management (Context API/Zustand)

**Phase 3: Core Features**
1. Build numerology form with validation
2. Create pyramid grid visualization
3. Implement root number analysis display
4. Build SWOT analysis component
5. Add business guidance features
6. Create learning style analysis

**Phase 4: User Management**
1. Implement authentication UI
2. Build user profile management
3. Add calculation history features
4. Implement save/delete calculations
5. Create dashboard for users

**Phase 5: Enhancement & Testing**
1. Add responsive design refinements
2. Implement error handling and loading states
3. Add form data persistence (localStorage)
4. Write comprehensive tests
5. Performance optimization
6. SEO optimization

**Phase 6: Deployment**
1. Configure production environment
2. Set up CI/CD pipeline
3. Deploy to staging environment
4. User acceptance testing
5. Production deployment

### 10. EXAMPLE CALCULATION TEST

**Test Case:**
- Name: "Ricky"
- Birthdate: "31-July-1985"
- Industry: "Sales"

**Expected Results:**
```
L5: [31, 7, 19, 85]
L6: [4, 7, 1, 4]
L7: [2, 5]  // reduce(4+7)=2, reduce(1+4)=5
L8: [9, 6, 9, 7, 6, 9, 6]
    // A=reduce(7+2)=9, B=reduce(4+2)=6, C=reduce(9+6)=6
    // D=reduce(1+5)=6, E=reduce(4+5)=9, F=reduce(6+9)=6
    // G=reduce(2+5)=7
    // L8=[C,B,A,G,D,E,F]=[6,6,9,7,6,9,6]
L9: [3, 6, 9]
    // Left=reduce(5+7)=3, Right=reduce(2+7)=9, Middle=reduce(3+9)=3
Root Number: 7 (G value)
```

**Validation Points:**
- Date parsing should handle "31-July-1985" format correctly
- Digit reduction function must work properly (19 -> 1+9 = 10 -> 1+0 = 1)
- Special case: Years ending in '00' should become 5 in L6
- SWOT combinations should only match official list
- Root number interpretation should return "KAWAN/PENGIKUT" data

### 11. ERROR HANDLING

**Backend Error Responses:**
```php
// Validation errors
{
    "success": false,
    "message": "Validation failed",
    "errors": {
        "birthdate": ["Invalid date format. Please use DD-MM-YYYY or DD Month YYYY"]
    }
}

// System errors
{
    "success": false,
    "message": "Internal server error",
    "error_code": "CALC_001"
}
```

**Frontend Error Handling:**
- Form validation with real-time feedback
- API error messages display
- Graceful fallbacks for missing data
- Retry mechanisms for failed requests
- User-friendly error messages in both English and Malay

### 12. MULTILINGUAL SUPPORT

**Language Structure:**
```php
// resources/lang/ms/numerology.php
return [
    'root_titles' => [
        1 => 'SENDIRI (Pemimpin Berdikari)',
        2 => 'SUARA (Suara/Harmoni)',
        // ... etc
    ],
    'elements' => [
        'LOGAM/BESI' => 'Logam/Besi',
        'AIR' => 'Air', 
        'API' => 'Api',
        'KAYU' => 'Kayu',
        'TANAH' => 'Tanah'
    ]
];

// resources/lang/en/numerology.php
return [
    'root_titles' => [
        1 => 'INDEPENDENT (Independent Leader)',
        2 => 'VOICE (Voice/Harmony)',
        // ... etc
    ]
];
```

### 13. PERFORMANCE CONSIDERATIONS

**Backend Optimization:**
- Cache root number interpretations
- Database indexing on frequently queried fields
- API response caching for static data
- Optimize SWOT combination algorithm
- Use database transactions for data integrity

**Frontend Optimization:**
- Lazy loading for components
- Memoization of calculation results
- Image optimization and lazy loading
- Code splitting by routes
- Service worker for offline functionality

### 14. SECURITY REQUIREMENTS

**Backend Security:**
- Input validation and sanitization
- SQL injection prevention (use Eloquent ORM)
- XSS protection
- CSRF protection
- Rate limiting on API endpoints
- Secure password hashing
- API authentication with tokens

**Frontend Security:**
- Secure token storage
- Input sanitization
- XSS prevention
- Secure HTTP headers
- Content Security Policy

### 15. DEPLOYMENT CONFIGURATION

**Environment Variables:**
```env
# .env
APP_NAME="SWOT Talent Numerology"
APP_ENV=production
APP_KEY=base64:...
APP_DEBUG=false
APP_URL=http://swottalent.test

DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=swottalent
DB_USERNAME=root
DB_PASSWORD=

SANCTUM_STATEFUL_DOMAINS=swottalent.test
SESSION_DOMAIN=.swottalent.test
```

**Laravel Herd Configuration:**
- Ensure project is accessible at http://swottalent.test
- Configure SSL for https if needed
- Set up proper database credentials
- Configure mail settings for notifications

---

## GETTING STARTED

1. **Clone/Setup Laravel Project:**
   ```bash
   composer create-project laravel/laravel swottalent
   cd swottalent
   composer install
   ```

2. **Configure Database:**
   ```bash
   php artisan migrate
   php artisan db:seed
   ```

3. **Install React Dependencies:**
   ```bash
   npm install react react-dom @vitejs/plugin-react
   npm install tailwindcss @tailwindcss/forms
   npm install axios react-router-dom react-hook-form
   ```

4. **Start Development:**
   ```bash
   php artisan serve
   npm run dev
   ```

Please start by setting up the basic Laravel project structure with the numerology calculation service, then move to the React frontend integration. Focus on accuracy of the mathematical calculations first, then build the UI components.

**Priority Order:**
1. ✅ Numerology calculation engine (backend)
2. ✅ Database schema and migrations
3. ✅ API endpoints and controllers
4. ✅ React form component
5. ✅ Pyramid grid visualization
6. ✅ Results display components
7. ✅ Authentication system
8. ✅ User dashboard and saved calculations