export interface ValidationResult {
  isValid: boolean;
  error?: string;
}

export function validateName(name: string): ValidationResult {
  if (!name || name.trim().length === 0) {
    return { isValid: false, error: 'Name is required' };
  }
  
  if (name.trim().length < 2) {
    return { isValid: false, error: 'Name must be at least 2 characters long' };
  }
  
  if (name.length > 255) {
    return { isValid: false, error: 'Name must be less than 255 characters' };
  }
  
  return { isValid: true };
}

export function validateDate(dateString: string): ValidationResult {
  if (!dateString || dateString.trim().length === 0) {
    return { isValid: false, error: 'Date is required' };
  }

  // Check common formats: DD-MM-YYYY, DD/MM/YYYY, DD Month YYYY, DD-Month-YYYY
  const patterns = [
    /^(\d{1,2})-(\d{1,2})-(\d{4})$/,     // DD-MM-YYYY
    /^(\d{1,2})\/(\d{1,2})\/(\d{4})$/,   // DD/MM/YYYY
    /^(\d{1,2})\s([A-Za-z]+)\s(\d{4})$/, // DD Month YYYY
    /^(\d{1,2})-([A-Za-z]+)-(\d{4})$/,   // DD-Month-YYYY
  ];

  const months = [
    'january', 'february', 'march', 'april', 'may', 'june',
    'july', 'august', 'september', 'october', 'november', 'december',
    'januari', 'februari', 'mac', 'april', 'mei', 'jun',
    'julai', 'ogos', 'september', 'oktober', 'november', 'disember'
  ];

  let isValidFormat = false;
  
  for (const pattern of patterns) {
    const match = dateString.match(pattern);
    if (match) {
      isValidFormat = true;
      const [, dayStr, monthStr, yearStr] = match;
      
      const day = parseInt(dayStr);
      const year = parseInt(yearStr);
      
      // Validate year range
      if (year < 1800 || year > 2099) {
        return { isValid: false, error: 'Year must be between 1800 and 2099' };
      }
      
      // If month is text, validate it's a known month
      if (isNaN(parseInt(monthStr))) {
        if (!months.includes(monthStr.toLowerCase())) {
          return { isValid: false, error: 'Unknown month name' };
        }
      } else {
        const month = parseInt(monthStr);
        if (month < 1 || month > 12) {
          return { isValid: false, error: 'Month must be between 1 and 12' };
        }
      }
      
      // Basic day validation
      if (day < 1 || day > 31) {
        return { isValid: false, error: 'Day must be between 1 and 31' };
      }
      
      break;
    }
  }
  
  if (!isValidFormat) {
    return { 
      isValid: false, 
      error: 'Invalid date format. Please use DD-MM-YYYY, DD/MM/YYYY, DD Month YYYY, or DD-Month-YYYY' 
    };
  }
  
  return { isValid: true };
}

export function validateEmail(email: string): ValidationResult {
  if (!email || email.trim().length === 0) {
    return { isValid: false, error: 'Email is required' };
  }
  
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  if (!emailRegex.test(email)) {
    return { isValid: false, error: 'Please enter a valid email address' };
  }
  
  return { isValid: true };
}

export function validatePassword(password: string): ValidationResult {
  if (!password || password.length === 0) {
    return { isValid: false, error: 'Password is required' };
  }
  
  if (password.length < 8) {
    return { isValid: false, error: 'Password must be at least 8 characters long' };
  }
  
  return { isValid: true };
}

export function validatePasswordConfirmation(password: string, confirmation: string): ValidationResult {
  if (password !== confirmation) {
    return { isValid: false, error: 'Passwords do not match' };
  }
  
  return { isValid: true };
}